'use client'

import { FadeIn } from "@/components/animations/fade-in";
import { CardHeader } from "@/components/ui/card";
import { Label } from "@/components/ui/label";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { Switch } from "@/components/ui/switch";
import dynamic from "next/dynamic";
import { useState } from "react";
import { useOtpLogs } from "../hooks/use-otp-logs";
import LoadingCircle from "@/components/icons/loading.circle";

const DynamicMap = dynamic(() => import('@/features/engagement/components/Map'), {
  ssr: false,
})

const animateClass: string = 'transition-color duration-300 font-bold'
interface DateRange {
  name: string, id: string, days?: number, months?: number
}

const dateRanges: DateRange[] = [
  { name: 'Last 7 days', id: '1', days: 7 },
  { name: 'Last 30 days', id: '2', days: 30 },
  { name: 'Last 6 months', id: '3', months: 6 },
  { name: 'Last 12 months', id: '4', months: 12 },
]

const formatDate = (date: Date): string => {
  const month = String(date.getMonth() + 1).padStart(2, '0');
  const day = String(date.getDate()).padStart(2, '0');
  const year = date.getFullYear();
  return `${month}-${day}-${year}`;
};

const calculateStartDate = (range: DateRange): string => {
  const startDate = new Date();

  if (range.days) {
    startDate.setDate(startDate.getDate() - range.days);
  } else if (range.months) {
    startDate.setMonth(startDate.getMonth() - range.months);
  }

  return formatDate(startDate);
};

const getDates = (dateRange: DateRange): { start: string; end: string } => {
  const start = calculateStartDate(dateRange);
  const end = formatDate(new Date());
  return { start, end };
};

export default function QueryAnalytics() {
  const [dateFilter, setDateFilter] = useState<DateRange>(dateRanges[1])
  const [showOrigin, setShowOrigin] = useState<boolean>(true)
  const [showDestination, setShowDestination] = useState<boolean>(false)

  const dateRange = getDates(dateFilter)
  const { data, isLoading, error } = useOtpLogs(dateRange.start, dateRange.end);

  return (
    <div className="m-1 grid auto-rows-min gap-3 md:grid-cols-3">
      <div className="flex flex-col gap-3">
        <FadeIn delay={0.7}>
          <CardHeader className="text-md font-semibold p-0">Query Analytics</CardHeader>
        </FadeIn>
        <FadeIn delay={0.75} className="items-center flex flex-row gap-3" >
          <Select
            value={dateFilter.id}
            onValueChange={(value: string) => {
              const dateRange = dateRanges.find(r => r.id === value)
              dateRange && setDateFilter(dateRange)
            }}>
            <SelectTrigger className="w-full border border-input">
              <SelectValue placeholder="Select a filter" />
            </SelectTrigger>
            <SelectContent>
              {dateRanges.map((r) =>
                <SelectItem key={r.id} value={r.id}>{r.name}</SelectItem>
              )}
            </SelectContent>
          </Select>
          <div className={animateClass} style={{ opacity: isLoading ? '1' : '0' }}>
            <LoadingCircle className="color-primary" />
          </div>
        </FadeIn>
        <FadeIn className="flex items-center space-x-2" delay={0.8}>
          <Switch id="origin-mode" checked={showOrigin}
            onCheckedChange={(e) => setShowOrigin(e)} />
          <Label htmlFor="origin-mode">Show <span className={animateClass} style={{ color: `${showOrigin ? 'green' : ''}` }}>Origin</span> Clusters</Label>
        </FadeIn>
        <FadeIn className="flex items-center space-x-2" delay={0.9}>
          <Switch id="destination-mode" checked={showDestination}
            onCheckedChange={(e) => setShowDestination(e)} />
          <Label htmlFor="destination-mode">Show <span className={animateClass} style={{ color: `${showDestination ? 'red' : ''}` }}>Destination</span> Clusters</Label>
        </FadeIn>
        {error && (
          <FadeIn className="flex items-center space-x-2" delay={0.9}>
            <div className="text-red-500">
              Failed to retrieve map data.
            </div>
          </FadeIn>
        )}
      </div>
      <FadeIn className="min-h-[85vh] flex-1 rounded-xl bg-muted/50 col-span-2" delay={1}>
        <DynamicMap data={data ?? []} showOrigin={showOrigin} showDestination={showDestination} />
      </FadeIn>
    </div>
  );
}